/*
 *  record.js
 *
 *  Copyright (c) 2018-2023 Claris International Inc.  All rights reserved.
 *  CLARIS CONFIDENTIAL - unpublished proprietary source code
 *
 */
'use strict';
var errorHandler = require('../routes/error');
var util = require('../routes/util');

var recordId;

/**
 * @api {GET} /databases/:solution/layout/:layout/record/:recordId Get Record By Id
 * @apiDescription Get a record for given recordId :record in specified solution and layout.
 * @apiName getRecord
 * @apiGroup Record
 *
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *       "result": "OK",
 *       "record": "{...}"
 *     }
 *
 * @apiUse BadRequestError
 * @apiUse UnauthorizedError
 * @apiUse FMServiceError
 *
 *
 * @apiSampleRequest /databases/:solution/layout/:layout/record/:recordId
 */
module.exports.getRecordById = function(req, res, next){
	var input = req.originalUrl.split('?');
	var ipStr = util.parseClientIp(req);
	var params = {
		'solution': req.swagger.params.database.value,
		'layout': req.swagger.params.layout.value,
		'requestMethod': req.method,
		'requestPath': req.originalUrl,
		'requestIp': ipStr,
		'inputBandwidth': (req.body) ? JSON.stringify(req.body).length.toString() : "0",
		'version': req.swagger.params.version.value
	};

	var result = util.setAuthentication(req, res, params);
	if(result) {
		if (typeof req.swagger.params.recordId != 'undefined') {
			if (isNaN(req.swagger.params.recordId.value) || req.swagger.params.recordId.value <= 0) {
				return errorHandler.handleError('BadRequest', req, res, 'recordId must be an integer with value greater than 0.',"960")			
			}
			params.recordId = req.swagger.params.recordId.value;
		}

		for (var propName in req.query) {
			// enforce that only supported query parameters are allowed
			if( (['portal','script','script.param','script.prerequest', 'script.prerequest.param','script.presort','script.presort.param','layout.response', 'dateformats'].indexOf(propName) === -1) &&
					!propName.toLowerCase().startsWith("_offset.") &&
					!propName.toLowerCase().startsWith("_limit.") )
				return errorHandler.handleError('BadRequest', req, res, 'Unknown parameter(s): '+propName,"960");

			if (propName.toLowerCase().startsWith("_offset") || propName.toLowerCase().startsWith("_limit")) {
				if (isNaN(req.query[propName]) || req.query[propName]<=0) {
					return errorHandler.handleError('BadRequest', req, res, 'The '+ propName +' must be an integer with value greater than 0.',"960")
				}
				params[propName.toLowerCase().substring(1)] = req.query[propName];
			} else if (propName.toLowerCase().startsWith("dateformats")) {
				if (isNaN(req.query[propName]) || req.query[propName]<0 || req.query[propName]>2) {
					return errorHandler.handleError('BadRequest', req, res, 'The '+ propName +' must be an integer with value 0, 1 or 2.',"960")
				}
				params[propName.toLowerCase()] = req.query[propName];
			} else {
				var val = req.query[propName];
				if (val instanceof Array) {
					return errorHandler.handleError('BadRequest', req, res, 'Request contains multiple '+ propName +' parameters',"960")
				} else {
					if (propName.charAt(0) == '_') {
						params[propName.toLowerCase().substring(1)] = req.query[propName];
					} else {
						params[propName.toLowerCase()] = req.query[propName];
					}
				}
			}
		}

		if (req.query.hasOwnProperty('script.param') && req.query['script.param'] && !req.query.hasOwnProperty('script')) {
			return errorHandler.handleError('BadRequest', req, res, "script.param without script","960");
		} else if (req.query.hasOwnProperty('script.prerequest.param') && req.query['script.prerequest.param'] && !req.query.hasOwnProperty('script.prerequest')) {
			return errorHandler.handleError('BadRequest', req, res, "script.prerequest.param without script.prerequest","960");
		} else if (req.query.hasOwnProperty('script.presort.param') && req.query['script.presort.param'] && !req.query.hasOwnProperty('script.presort')) {
			return errorHandler.handleError('BadRequest', req, res, "script.presort.param without script.presort","960");
		}

		try{
			util.thrift_client.getRecords(params,
		function(thrifError, thrifResult){
				return util.handleThrifReturn(thrifError, thrifResult, req, res);
			});
		} catch (err){
			util.thriftExceptionHandler(err, req, res);
		}
	}
};
/**
 * @api {POST} /databases/:solution/layout/:layout/record/:recordId  Duplicate Record
 * @apiDescription Duplcates a record for given recordId in specified solution layout.
 * @apiName duplicateRecord
 * @apiGroup Record
 *
 * @apiSuccess {String} errorCode "0"
 *
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *       "result": "OK",
 *       "recordId": "recordId"
 *     }
 *
 * @apiUse BadRequestError
 * @apiUse UnauthorizedError
 * @apiUse FMServiceError
 *
 *
 * @apiSampleRequest /databases/:solution/layout/:layout/record/:recordId
 */
module.exports.duplicate = function(req, res, next) {
	var ipStr = util.parseClientIp(req);
	var params = {
		'solution': req.swagger.params.database.value,
		'layout': req.swagger.params.layout.value,
		'recordId': req.swagger.params.recordId.value,
		'requestMethod': req.method,
		'requestPath': req.originalUrl,
		'requestIp': ipStr,
		'inputBandwidth': (req.body) ? JSON.stringify(req.body).length.toString() : "0",
		'version': req.swagger.params.version.value
	};

	var result = util.setAuthentication(req, res, params);	
	if(result) {
		if (isNaN(params.recordId) || params.recordId <= 0) {
			return errorHandler.handleError("BadRequest", req, res, "recordId must be an integer with value than 0" ,"960");					
		}
		if (req.body.hasOwnProperty('script') && req.body['script']) {
			if (typeof req.body['script'] !== "string" && !(req.body['script'] instanceof String)) {
				return errorHandler.handleError("BadRequest", req, res, "script value is not a string" ,"960");
			}
			params['script'] = req.body['script'];
			delete req.body['script'];
			if (req.body.hasOwnProperty('script.param') && req.body['script.param']) {
				if (typeof req.body['script.param'] !== "string" && !(req.body['script.param'] instanceof String)) {
					return errorHandler.handleError("BadRequest", req, res, "script.param value is not a string" ,"960");
				}
				params['script.param'] = req.body['script.param'];
				delete req.body['script.param'];
			}
		}
		else if (req.body.hasOwnProperty('script.param')) {
			return errorHandler.handleError("BadRequest", req, res, "script.param without script" ,"960");
		}
		if (req.body.hasOwnProperty('script.prerequest') && req.body['script.prerequest']) {
			if (typeof req.body['script.prerequest'] !== "string" && !(req.body['script.prerequest'] instanceof String)) {
				return errorHandler.handleError("BadRequest", req, res, "script.prerequest value is not a string" ,"960");
			}
			params['script.prerequest'] = req.body['script.prerequest'];
			delete req.body['script.prerequest'];
			if (req.body['script.prerequest.param']) {
				if (typeof req.body['script.prerequest.param'] !== "string" && !(req.body['script.prerequest.param'] instanceof String)) {
					return errorHandler.handleError("BadRequest", req, res, "script.prerequest.param value is not a string" ,"960");
				}
				params['script.prerequest.param'] = req.body['script.prerequest.param'];
				delete req.body['script.prerequest.param'];
			}
		}
		else if (req.body.hasOwnProperty('script.prerequest.param')) {
			return errorHandler.handleError("BadRequest", req, res, "script.prerequest.param without script.prerequest" ,"960");		
		}
		if (req.body.hasOwnProperty('script.presort') && req.body['script.presort']) {
			if (typeof req.body['script.presort'] !== "string" && !(req.body['script.presort'] instanceof String)) {
				return errorHandler.handleError("BadRequest", req, res, "script.presort value is not a string" ,"960");
			}
			params['script.presort'] = req.body['script.presort'];
			delete req.body['script.presort'];
			if (req.body['script.presort.param']) {
				if (typeof req.body['script.presort.param'] !== "string" && !(req.body['script.presort.param'] instanceof String)) {
					return errorHandler.handleError("BadRequest", req, res, "script.presort.param value is not a string" ,"960");
				}
				params['script.presort.param'] = req.body['script.presort.param'];
				delete req.body['script.presort.param'];
			}
		}
		else if (req.body.hasOwnProperty('script.presort.param')) {
			return errorHandler.handleError("BadRequest", req, res, "script.presort.param without script.presort" ,"960");		
		}

		var remainingProps = Object.keys(req.body);
		if (remainingProps.length > 0) {
			return errorHandler.handleError("BadRequest", req, res, "Unknown parameter(s): "+remainingProps.join(','),"960");		
		}

		try{
			util.thrift_client.duplicateRecord(params,
				function(thrifError, thrifResult){
					return util.handleThrifReturn(thrifError, thrifResult, req, res);
			});
		} catch (err){
			util.thriftExceptionHandler(err, req, res);
		}
	}

};

/**
 * @api {delete} /databases/:solution/layout/:layout/record/:recordId Delete Record
 * @apiDescription Delete a record for given recordId in specified solution layout.
 * @apiName deleteRecord
 * @apiGroup Record
 *
 * @apiSuccess {String} errorCode "0"
 *
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *       "result": "OK",
 *       "recordId": "recordId"
 *     }
 *
 * @apiUse BadRequestError
 * @apiUse UnauthorizedError
 * @apiUse FMServiceError
 *
 *
 * @apiSampleRequest /databases/:solution/layout/:layout/record/:recordId
 */
module.exports.delete = function(req, res, next){
	try{
		var ipStr = util.parseClientIp(req);
		var params = {
			'solution': req.swagger.params.database.value,
			'layout': req.swagger.params.layout.value,
			'recordId': req.swagger.params.recordId.value,
			'requestMethod': req.method,
			'requestPath': req.originalUrl,
			'requestIp': ipStr,
			'version': req.swagger.params.version.value
		}
		var result = util.setAuthentication(req, res, params);
		if(result){
			// enforce only supported parameters
			if (req.query.hasOwnProperty('script') && req.query['script']) {
				params['script'] = req.query['script'];
				delete req.query['script'];
				if (req.query.hasOwnProperty('script.param') && req.query['script.param']) {
					params['script.param'] = req.query['script.param'];
					delete req.query['script.param'];
				}
			} else if (req.query.hasOwnProperty('script.param')) {
				return errorHandler.handleError("BadRequest", req, res, "script.param without script","960");
			}
			if (req.query.hasOwnProperty('script.prerequest') && req.query['script.prerequest']) {
				params['script.prerequest'] = req.query['script.prerequest'];
				delete req.query['script.prerequest'];
				if (req.query['script.prerequest.param']) {
					params['script.prerequest.param'] = req.query['script.prerequest.param'];
					delete req.query['script.prerequest.param'];
				}
			}
			else if (req.query.hasOwnProperty('script.prerequest.param')) {
				return errorHandler.handleError("BadRequest", req, res, "script.prerequest.param without script.prerequest","960");
			}
			if (req.query.hasOwnProperty('script.presort') && req.query['script.presort']) {
				params['script.presort'] = req.query['script.presort'];
				delete req.query['script.presort'];
				if (req.query['script.presort.param']) {
					params['script.presort.param'] = req.query['script.presort.param'];
					delete req.query['script.presort.param'];
				}
			}
			else if (req.query.hasOwnProperty('script.presort.param')) {
				return errorHandler.handleError("BadRequest", req, res, "script.presort.param without script.presort","960");
			}

			var remainingProps = Object.keys(req.query);
			if (remainingProps.length > 0) {
				return errorHandler.handleError("BadRequest", req, res, "Unknown parameter(s): "+remainingProps.join(','),"960");		
			}
				
			util.thrift_client.deleteRecord(
				params,
				function(thrifError, thrifResult){
					return util.handleThrifReturn(thrifError, thrifResult, req, res);
				}
			);
		}
	} catch (err){
		util.thriftExceptionHandler(err, req, res);
	}
};

/**
 * @api {PATCH} /databases/:solution/layout/:layout/record/:recordId  Edit Record
 * @apiDescription Update a record for given recordId in specified solution layout.
 * @apiName editRecord
 * @apiGroup Record
 *
 * @apiUse ContentTypeHeader
 * @apiUse AccessTokenHeader
 *
 * @apiParam {json} data	A stringify JSON object contains field-value pairs in the target layout.
 * Only specified field-values in data will be modified.
 * Related records in the layout or portol can be created, updated and deleted with syntas as shown in the example.
 * If "{}" is provided as data value, the target record won't be updated.
 *
 * @apiParam {string} [modId]	Optional Id for last modification, which can be used it to verify if the cached record is up-to-data one.
 * modId is only supported for main table records, NOT applicable for portal/related rows.
 *
 * @apiParamExample {json} Request-Body-Example:
 *    {"data": {
 *       "field_1": "value_1",
 *       "field_2": "value_2",
 *       "repetitionField(2)" : "fieldValue",		//update the 2nd item in repitition field
 *       "Orders::OrderDate.2":"12/20/2015",		//update OrderDate for related "Orders" record with recordId=2
 *       "Orders::OrderDate.0":"12/22/2015",		//create a related "Orders" record with specify OrderDate, (use 0 for recordId to create)
 *       "deleteRelated": "Orders.2"						//delete related "Orders" record with recordId=2
 *     },
 *     modId: "3"   //Id for last modification is 3 for my cached record, allow server to check it is up-to-data
 *    }
 *
 *
 * @apiSuccess {String} errorCode "0"
 *
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *       "result": "OK",
 *       "recordId": "recordId"
 *     }
 *
 * @apiUse BadRequestError
 * @apiUse UnauthorizedError
 * @apiUse FMServiceError
 *
 *
 * @apiSampleRequest /databases/:solution/layout/:layout/record/:recordId
 */
module.exports.edit = function(req, res, next){
	var data;
	var portalData;
	if (!req.body['fieldData'] && !req.body['portalData']) {
		return errorHandler.handleError("BadRequest", req, res, "Neither 'fieldData' nor 'portalData' was specified in the request.","10");		
	}
	if (req.body['fieldData']) {
		data = (typeof req.body['fieldData'] === "object") ? JSON.stringify(req.body['fieldData']) : req.body['fieldData'].toString();
		if(data.trim().length===0) {
			return errorHandler.handleError("BadRequest", req, res, "fieldData is empty.","10");
		}
		if (typeof JSON.parse(data) !== "object") {
			return errorHandler.handleError("BadRequest", req, res, "fieldData is not an object","960");
		}
		delete req.body['fieldData'];
	}
	if (req.body.hasOwnProperty('portalData')) {
		if (req.body['portalData']) {
			if (req.body['portalData'] !== Object(req.body['portalData'])) {
				return errorHandler.handleError("BadRequest", req, res, "portalData is not an object","960");					
			}
			for (var portalName in req.body['portalData']) {
				var portalRowArray = req.body['portalData'][portalName];
				if (!Array.isArray(portalRowArray)) {
					return errorHandler.handleError("BadRequest", req, res, "'"+ portalName +"' in portalData is not an array","960");					
				}
			}
			portalData = (typeof req.body['portalData'] != "string") ? JSON.stringify(req.body['portalData']) : req.body['portalData'];
			if(portalData.trim().length===0) {
				return errorHandler.handleError("BadRequest", req, res, "portalData is empty.","10");
			}
			delete req.body['portalData'];
		} else {
			return errorHandler.handleError("BadRequest", req, res, "portalData is empty.","10");
		}
	}

	if (!data) {
		data = "";
	}
	var ipStr = util.parseClientIp(req);
	var params = {
		'solution': req.swagger.params.database.value,
		'layout': req.swagger.params.layout.value,
		'recordId': req.swagger.params.recordId.value,
		'fieldData': data,
		'requestMethod': req.method,
		'requestPath': req.originalUrl,
		'requestIp': ipStr,
		'inputBandwidth': (req.body) ? JSON.stringify(req.body).length.toString() : "0",
		'version': req.swagger.params.version.value,
		'options': data
	};

	var result = util.setAuthentication(req, res, params);
	if(result){
		if (portalData) {
			params.portalData = portalData;
		}
		if (isNaN(params.recordId) || params.recordId <= 0) {
			return errorHandler.handleError("BadRequest", req, res, "recordId must be an integer with value than 0" ,"960");					
		}
		if(req.body.hasOwnProperty('options')){
			params.options = JSON.stringify(req.body['options']); 
			delete req.body['options'];
		}
		if(req.body.hasOwnProperty('modId')){
			if (typeof req.body['modId'] === 'boolean' || isNaN(req.body['modId']) || req.body['modId'] < 0) {
				return errorHandler.handleError("BadRequest", req, res, "modId must be an integer with value greater than or equal to 0" ,"960");			
			} else if (typeof req.body['modId'] === 'string' || req.body['modId'] instanceof String) {
				if (req.body['modId'].length == 0) {
					return errorHandler.handleError("BadRequest", req, res, "modId is an empty string" ,"960");
				}
			}
			params.modId = req.body['modId'].toString();
			delete req.body['modId'];
		}
		if (req.body.hasOwnProperty('script') && req.body['script']) {
			if (typeof req.body['script'] !== "string" && !(req.body['script'] instanceof String)) {
				return errorHandler.handleError("BadRequest", req, res, "script value is not a string" ,"960");
			}
			params['script'] = req.body['script'];
			delete req.body['script'];
			if (req.body.hasOwnProperty('script.param') && req.body['script.param']) {
				if (typeof req.body['script.param'] !== "string" && !(req.body['script.param'] instanceof String)) {
					return errorHandler.handleError("BadRequest", req, res, "script.param value is not a string" ,"960");
				}
				params['script.param'] = req.body['script.param'];
				delete req.body['script.param'];
			}
		}
		else if (req.body.hasOwnProperty('script.param')) {
			return errorHandler.handleError("BadRequest", req, res, "script.param without script" ,"960");
		}
		if (req.body.hasOwnProperty('script.prerequest') && req.body['script.prerequest']) {
			if (typeof req.body['script.prerequest'] !== "string" && !(req.body['script.prerequest'] instanceof String)) {
				return errorHandler.handleError("BadRequest", req, res, "script.prerequest value is not a string" ,"960");
			}
			params['script.prerequest'] = req.body['script.prerequest'];
			delete req.body['script.prerequest'];
			if (req.body['script.prerequest.param']) {
				if (typeof req.body['script.prerequest.param'] !== "string" && !(req.body['script.prerequest.param'] instanceof String)) {
					return errorHandler.handleError("BadRequest", req, res, "script.prerequest.param value is not a string" ,"960");
				}
				params['script.prerequest.param'] = req.body['script.prerequest.param'];
				delete req.body['script.prerequest.param'];
			}
		}
		else if (req.body.hasOwnProperty('script.prerequest.param')) {
			return errorHandler.handleError("BadRequest", req, res, "script.prerequest.param without script.prerequest" ,"960");		
		}
		if (req.body.hasOwnProperty('script.presort') && req.body['script.presort']) {
			if (typeof req.body['script.presort'] !== "string" && !(req.body['script.presort'] instanceof String)) {
				return errorHandler.handleError("BadRequest", req, res, "script.presort value is not a string" ,"960");
			}
			params['script.presort'] = req.body['script.presort'];
			delete req.body['script.presort'];
			if (req.body['script.presort.param']) {
				if (typeof req.body['script.presort.param'] !== "string" && !(req.body['script.presort.param'] instanceof String)) {
					return errorHandler.handleError("BadRequest", req, res, "script.presort.param value is not a string" ,"960");
				}
				params['script.presort.param'] = req.body['script.presort.param'];
				delete req.body['script.presort.param'];
			}
		}
		else if (req.body.hasOwnProperty('script.presort.param')) {
			return errorHandler.handleError("BadRequest", req, res, "script.presort.param without script.presort" ,"960");		
		}
		if (req.body.hasOwnProperty('dateformats')) {
			params['dateformats'] = "" + req.body['dateformats'];
			delete req.body['dateformats'];
		}

		var remainingProps = Object.keys(req.body);
		if (remainingProps.length > 0) {
			return errorHandler.handleError("BadRequest", req, res, "Unknown parameter(s): "+remainingProps.join(','),"960");		
		}

		try{
			util.thrift_client.editRecord(params,
				function(thrifError, thrifResult){
					return util.handleThrifReturn(thrifError, thrifResult, req, res);
			});
		} catch (err){
			util.thriftExceptionHandler(err, req, res);
		}
	}
};
